package com.tykj.classified.controller;

import com.alibaba.fastjson.JSONObject;
import com.deepoove.poi.data.TextRenderData;
import com.deepoove.poi.data.style.Style;
import com.tykj.classified.config.exception.BizException;
import com.tykj.classified.dao.db1.ConfClassifiedBasisDao;
import com.tykj.classified.dao.db1.ConfManageDetailDao;
import com.tykj.classified.dao.db1.EmployeeDao;
import com.tykj.classified.entity.db1.ConfManageDetail;
import com.tykj.classified.entity.db1.Employee;
import com.tykj.classified.entity.vo.*;
import com.tykj.classified.response.ServerResponse;
import com.tykj.classified.service.ConfClassifiedDecryptService;
import com.tykj.classified.utils.freemake.FreeMakerUtils;
import com.tykj.classified.utils.idwork.IdWorker;
import com.tykj.classified.utils.map.MapUtils;
import com.tykj.classified.utils.string.StringUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author sjq
 */
@Slf4j
@RestController
@RequestMapping("/confClassifiedDecrypt")
@Api(tags = "定密解密接口相关文档")
public class ConfClassifiedDecryptController {

    private final ConfClassifiedDecryptService confClassifiedDecryptService;

    @Value("${file.upload.path}")
    private String path;

    private final ConfManageDetailDao confManageDetailDao;

    private final EmployeeDao employeeDao;

    private final ConfClassifiedBasisDao confClassifiedBasisDao;

    private final IdWorker idWorker;

    public ConfClassifiedDecryptController(ConfClassifiedDecryptService confClassifiedDecryptService, ConfManageDetailDao confManageDetailDao, EmployeeDao employeeDao, ConfClassifiedBasisDao confClassifiedBasisDao, IdWorker idWorker) {
        this.confClassifiedDecryptService = confClassifiedDecryptService;
        this.confManageDetailDao = confManageDetailDao;
        this.employeeDao = employeeDao;
        this.confClassifiedBasisDao = confClassifiedBasisDao;
        this.idWorker = idWorker;
    }

    /**
     * title：点击解密时根据定密id查询基本信息以及解密信息
     * action：根据定密id查询定密基本信息和解密信息，解密可能有多次
     * @param classifiedId 定密id
     * @return ConfClassifiedAndDecryptVo 定密信息和解密信息Vo
     */
    @ApiOperation("根据定密id查询基本信息以及解密信息")
    @GetMapping("/findClassifiedDecryptByClassifiedId")
    public ServerResponse<ConfClassifiedAndDecryptVo> findClassifiedDecryptByClassifiedId(@RequestParam("classifiedId") String classifiedId) {
        return ServerResponse.ok(confClassifiedDecryptService.findClassifiedDecryptByClassifiedId(classifiedId));
    }

    /**
     * title：点击解密时先调用这个接口查询是否有还未执行的解密操作
     * action：根据定密id查询是否存在已审核通过但未执行的解密操作
     * @param classifiedId 定密id
     * @return String
     */
    @ApiOperation("操作提示-解密新增")
    @GetMapping("/decryptNewOperations")
    public ServerResponse<String> decryptNewOperations(@RequestParam("classifiedId") String classifiedId) {
        confClassifiedDecryptService.decryptNewOperations(classifiedId);
        return ServerResponse.ok("新增成功");
    }

    /**
     * title：解密信息保存按钮调用接口
     * action：保存解密信息但未审核并且未执行，在下一次进来可以看到
     * @param confClassifiedDecryptVo 定密事项解密Vo
     * @return String
     */
    @ApiOperation("保存解密信息")
    @PostMapping("/addClassifiedDecrypt")
    public ServerResponse<String> addClassifiedDecrypt(@RequestBody ConfClassifiedDecryptVo confClassifiedDecryptVo) {
        confClassifiedDecryptService.addClassifiedDecrypt(confClassifiedDecryptVo);
        return ServerResponse.ok("新增解密信息成功");
    }

    /**
     * title：解密审核通过按钮调用接口
     * action：审核通过将解密信息的审核状态改为已审核，根据建议解密时间判断是否执行
     * 如果建议解密时间在今天或者之前，修改解密信息的执行状态为已执行，定密信息按照不同的解密信息进行变更
     * 1.如果是维持原定密决定，定密信息不做变更
     * 2.如果是变更，将定密信息的密级，保密期限，知悉范围改为变更后的信息
     * 3.如果是解密，a：解密并公开，将定密信息的状态改为已公开。b:解密后内部使用。将定密信息的状态改为工作状态。c:部分解密，将定密信息的状态改为部分解密。其他内容不变
     * 如果建议解密时间在之后，那么定密信息不做变更，通过解密的定时器去处理
     * @param confClassifiedDecryptVo 定密事项解密Vo
     * @return String
     */
    @ApiOperation("审核通过解密")
    @PostMapping("/decryptAuditPass")
    public ServerResponse<String> decryptAuditPass(@RequestBody ConfClassifiedDecryptVo confClassifiedDecryptVo) {
        confClassifiedDecryptService.decryptAuditPass(confClassifiedDecryptVo);
        return ServerResponse.ok("审核通过成功");
    }

    /**
     * title：点批量解密按钮时先调用此接口
     * action：当这些定密id里面如果有一个定密有已审核但未执行的解密操作，则抛出异常
     * @param confClassifiedDecryptBatchVo 定密id的List
     * @return String
     */
    @ApiOperation("操作提示-批量解密")
    @PostMapping("/decryptBatchOperations")
    public ServerResponse<String> decryptBatchOperations(@RequestBody ConfClassifiedDecryptBatchVo confClassifiedDecryptBatchVo) {
        log.info("批量解密操作提示=======================================================");
        log.info("classifiedIdList:{}", JSONObject.toJSONString(confClassifiedDecryptBatchVo.getClassifiedIdList()));
        confClassifiedDecryptService.decryptBatchOperations(confClassifiedDecryptBatchVo);
        return ServerResponse.ok("批量操作成功");
    }

    /**
     * title：批量解密审核通过按钮调用接口
     * action：给这些定密都新增一条解密信息记录，然后定密信息变更和解密操作一致
     * @param confClassifiedIdListAndDecryptVo 定密IdList和解密Vo
     * @return String
     */
    @ApiOperation("批量解密")
    @PostMapping("/batchDecrypt")
    public ServerResponse<String> batchDecrypt(@RequestBody ConfClassifiedIdListAndDecryptVo confClassifiedIdListAndDecryptVo) {
        log.info("批量解密=======================================================");
        log.info("confClassifiedIdListAndDecryptVo:{}",confClassifiedIdListAndDecryptVo);
        confClassifiedDecryptService.batchDecrypt(confClassifiedIdListAndDecryptVo);
        return ServerResponse.ok("批量解密成功");
    }

    /**
     * title：公开保存按钮调用接口
     * action：新增公开信息记录审核状态为未审核，执行状态为未执行
     * @param confClassifiedPublicVo 定密公开Vo
     * @return String
     */
    @ApiOperation("保存公开信息")
    @PostMapping("/addClassifiedPublic")
    public ServerResponse<String> addClassifiedPublic(@RequestBody ConfClassifiedPublicVo confClassifiedPublicVo) {
        log.info("新增or修改公开信息==============================================================");
        log.info("confClassifiedPublicVo:{}",confClassifiedPublicVo);
        confClassifiedDecryptService.addClassifiedPublic(confClassifiedPublicVo);
        return ServerResponse.ok("新增公开信息成功");
    }

    /**
     * title：根据定密id查询公开信息
     * action：根据定密id查询公开信息，如果有就返回，没有就new一个
     * @param classifiedId 定密id
     * @return ConfClassifiedPublicVo 定密公开Vo
     */
    @ApiOperation("根据定密id查询公开信息")
    @GetMapping("/findClassifiedPublicByClassifiedId")
    public ServerResponse<ConfClassifiedPublicVo> findClassifiedPublicByClassifiedId(@RequestParam("classifiedId") String classifiedId) {
        log.info("根据定密id查询公开信息==============================================================");
        log.info("classifiedId:{}",classifiedId);
        return ServerResponse.ok(confClassifiedDecryptService.findClassifiedPublicByClassifiedId(classifiedId));
    }

    /**
     * title：公开审核通过按钮调用接口
     * action：审核通过将公开信息的审核状态改为已审核，然后根据建议公开时间判断
     * 如果建议公开时间在今天或者之前，将公开信息的执行状态改为已执行，再把定密信息的状态改为已公开
     * 如果建议公开时间在之后，不做任何操作，由公开的定时器处理
     * @param confClassifiedPublicVo 定密公开Vo
     * @return String
     */
    @ApiOperation("审核通过公开")
    @PostMapping("/publicAuditPass")
    public ServerResponse<String> publicAuditPass(@RequestBody ConfClassifiedPublicVo confClassifiedPublicVo) {
        log.info("公开审核通过==============================================================");
        log.info("confClassifiedPublicVo:{}",confClassifiedPublicVo);
        confClassifiedDecryptService.publicAuditPass(confClassifiedPublicVo);
        return ServerResponse.ok("审核通过成功");
    }

    /**
     * title：定密解密预览调用接口
     * action：先将id转化成名称然后生成doc和pdf，并将pdf的所在路径返回
     * @param confClassifiedDecryptPreviewVo 定密解密预览Vo
     * @return String
     */
    @ApiOperation("定密解密预览")
    @PostMapping("/classifiedDecryptPreview")
    public ServerResponse<String> classifiedDecryptPreview(@RequestBody ConfClassifiedDecryptPreviewVo confClassifiedDecryptPreviewVo) {
        log.info("定密解密预览============================================================");
        log.info("confClassifiedDecryptPreviewVo:{}",confClassifiedDecryptPreviewVo);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        // 参数配置
        List<ConfManageDetail> confManageDetailList = confManageDetailDao.findAll();
        if (confClassifiedDecryptPreviewVo.getIssueDate() != null) {
            confClassifiedDecryptPreviewVo.setToIssueDate(sdf.format(confClassifiedDecryptPreviewVo.getIssueDate()));
        } else {
            throw new BizException("签发日期为空，无法预览");
        }
        if (confClassifiedDecryptPreviewVo.getBeforeClassification() != null) { // 前密级
            List<ConfManageDetail> collect = confManageDetailList.stream()
                    .filter(confManageDetail -> confClassifiedDecryptPreviewVo.getBeforeClassification() != null && confManageDetail.getId().equals(confClassifiedDecryptPreviewVo.getBeforeClassification()))
                    .collect(Collectors.toList());
            if (collect.size() > 0) {
                confClassifiedDecryptPreviewVo.setBeforeClassification(collect.get(0).getManageOptions());
            }
        }
        if (confClassifiedDecryptPreviewVo.getBeforeSecrecyTermType() != null) {
            if (confClassifiedDecryptPreviewVo.getBeforeSecrecyTermType() == 1) { // 保密期限类型：年
                confClassifiedDecryptPreviewVo.setBeforeSecrecyTermContent(confClassifiedDecryptPreviewVo.getBeforeSecrecyTermContent() + "年");
            } else if (confClassifiedDecryptPreviewVo.getBeforeSecrecyTermType() == 2) { // 保密期限类型：月
                confClassifiedDecryptPreviewVo.setBeforeSecrecyTermContent(confClassifiedDecryptPreviewVo.getBeforeSecrecyTermContent() + "月");
            } else if (confClassifiedDecryptPreviewVo.getBeforeSecrecyTermType() == 3) { // 保密期限类型：日
                confClassifiedDecryptPreviewVo.setBeforeSecrecyTermContent(confClassifiedDecryptPreviewVo.getBeforeSecrecyTermContent() + "日");
            }
        }
        if (confClassifiedDecryptPreviewVo.getAfterClassification() != null) { // 后密级
            List<ConfManageDetail> collect = confManageDetailList.stream()
                    .filter(confManageDetail -> confClassifiedDecryptPreviewVo.getAfterClassification() != null && confManageDetail.getId().equals(confClassifiedDecryptPreviewVo.getAfterClassification()))
                    .collect(Collectors.toList());
            if (collect.size() > 0) {
                confClassifiedDecryptPreviewVo.setAfterClassification(collect.get(0).getManageOptions());
            }
        }
        if (confClassifiedDecryptPreviewVo.getAfterSecrecyTermType() != null) {
            if (confClassifiedDecryptPreviewVo.getAfterSecrecyTermType() == 1) { // 保密期限类型：年
                confClassifiedDecryptPreviewVo.setAfterSecrecyTermContent(confClassifiedDecryptPreviewVo.getAfterSecrecyTermContent() + "年");
            } else if (confClassifiedDecryptPreviewVo.getAfterSecrecyTermType() == 2) { // 保密期限类型：月
                confClassifiedDecryptPreviewVo.setAfterSecrecyTermContent(confClassifiedDecryptPreviewVo.getAfterSecrecyTermContent() + "月");
            } else if (confClassifiedDecryptPreviewVo.getAfterSecrecyTermType() == 3) { // 保密期限类型：日
                confClassifiedDecryptPreviewVo.setAfterSecrecyTermContent(confClassifiedDecryptPreviewVo.getAfterSecrecyTermContent() + "日");
            }
        }
        Map map = MapUtils.beanToMap(confClassifiedDecryptPreviewVo);
        if (confClassifiedDecryptPreviewVo.getDecryptTime() != null) { // 建议解密时间
            confClassifiedDecryptPreviewVo.setToDecryptTime(sdf.format(confClassifiedDecryptPreviewVo.getDecryptTime()));
            map.put("toDecryptTime",confClassifiedDecryptPreviewVo.getToDecryptTime());
        }
        // wc-维持原定密决定,bg-变更,jmgk-解密并公开,jmnb-解密后内部使用,bfjm-部分解密
        if (confClassifiedDecryptPreviewVo.getDecryptType() == 1) { // 维持原定密决定
            map.put("wc",new TextRenderData("R",new Style("Wingdings 2",14)));
            map.put("bg",new TextRenderData("□",new Style("宋体",14)));
            map.put("jmgk",new TextRenderData("□",new Style("宋体",14)));
            map.put("jmnb",new TextRenderData("□",new Style("宋体",14)));
            map.put("bfjm",new TextRenderData("□",new Style("宋体",14)));
        } else if (confClassifiedDecryptPreviewVo.getDecryptType() == 2) { // 变更
            map.put("wc",new TextRenderData("□",new Style("宋体",14)));
            map.put("bg",new TextRenderData("R",new Style("Wingdings 2",14)));
            map.put("jmgk",new TextRenderData("□",new Style("宋体",14)));
            map.put("jmnb",new TextRenderData("□",new Style("宋体",14)));
            map.put("bfjm",new TextRenderData("□",new Style("宋体",14)));
        } else { // 解密
            if (confClassifiedDecryptPreviewVo.getDecryptMethod() == 1) { // 解密并公开
                map.put("wc",new TextRenderData("□",new Style("宋体",14)));
                map.put("bg",new TextRenderData("□",new Style("宋体",14)));
                map.put("jmgk",new TextRenderData("R",new Style("Wingdings 2",14)));
                map.put("jmnb",new TextRenderData("□",new Style("宋体",14)));
                map.put("bfjm",new TextRenderData("□",new Style("宋体",14)));
            } else if (confClassifiedDecryptPreviewVo.getDecryptMethod() == 2) { // 解密后内部使用
                map.put("wc",new TextRenderData("□",new Style("宋体",14)));
                map.put("bg",new TextRenderData("□",new Style("宋体",14)));
                map.put("jmgk",new TextRenderData("□",new Style("宋体",14)));
                map.put("jmnb",new TextRenderData("R",new Style("Wingdings 2",14)));
                map.put("bfjm",new TextRenderData("□",new Style("宋体",14)));
            } else { // 部分解密
                map.put("wc",new TextRenderData("□",new Style("宋体",14)));
                map.put("bg",new TextRenderData("□",new Style("宋体",14)));
                map.put("jmgk",new TextRenderData("□",new Style("宋体",14)));
                map.put("jmnb",new TextRenderData("□",new Style("宋体",14)));
                map.put("bfjm",new TextRenderData("R",new Style("Wingdings 2",14)));
            }
        }
        String word = new FreeMakerUtils().createWord(path,7, map, idWorker.nextId() + "");
        if (StringUtils.isEmpty(word)) {
            throw new BizException("生成文件失败");
        }
        return ServerResponse.ok(word);
    }

//    @ApiOperation("定密公开预览")
//    @PostMapping("/classifiedPublicPreview")
//    public ServerResponse<String> classifiedPublicPreview(@RequestBody ConfClassifiedDecryptPreviewVo confClassifiedDecryptPreviewVo) {
//        log.info("定密公开预览============================================================");
//        log.info("confClassifiedDecryptPreviewVo:{}",confClassifiedDecryptPreviewVo);
//        Map map = MapUtils.beanToMap(confClassifiedDecryptPreviewVo);
//        String word = new FreeMakerUtils().createWord(path,7, map, idWorker.nextId() + "");
//        if (StringUtils.isEmpty(word)) {
//            throw new BizException("生成文件失败");
//        }
//        return ServerResponse.ok(word);
//    }
}
