package com.tykj.classified.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.github.wenhao.jpa.PredicateBuilder;
import com.github.wenhao.jpa.Specifications;
import com.tykj.classified.config.exception.BizException;
import com.tykj.classified.dao.db1.ConfClassifiedBasisDao;
import com.tykj.classified.dao.db1.ConfUnitNodeDao;
import com.tykj.classified.dao.db1.DataUnitDao;
import com.tykj.classified.dao.db1.UnitBasisDao;
import com.tykj.classified.entity.db1.ConfClassifiedBasis;
import com.tykj.classified.entity.db1.ConfUnitNode;
import com.tykj.classified.entity.db1.DataUnit;
import com.tykj.classified.entity.db1.UnitBasis;
import com.tykj.classified.entity.vo.*;
import com.tykj.classified.entity.vo.mythreadlocal.MyThreadLocal;
import com.tykj.classified.service.DataUnitService;
import com.tykj.classified.utils.idwork.IdWorker;
import com.tykj.classified.utils.string.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author zhaolu
 * @create 2022-11-14 16:07
 */

@Service
public class DataUnitServiceImpl implements DataUnitService {

    @Autowired
    private DataUnitDao dataUnitDao;

    @Autowired
    private IdWorker idWorker;

    @Autowired
    private ConfUnitNodeDao confUnitNodeDao;

    @Value("${unit.firstNode}")
    private String unitFirstNode;

    @Autowired
    private ConfClassifiedBasisDao confClassifiedBasisDao;

    @Autowired
    private UnitBasisDao unitBasisDao;

    @Override
    public DataUnitShowVo findAllUnit(DataUnitPageVo dataUnitPageVo) {
        DataUnitShowVo dataUnitShowVo = new DataUnitShowVo();
        PredicateBuilder<DataUnit> predicateBuilder = Specifications.and();
        predicateBuilder.like(StringUtils.isNotEmpty(dataUnitPageVo.getUnitName()),"unitName","%" + dataUnitPageVo.getUnitName() + "%");
        if (!dataUnitPageVo.getUnitType().equals("全部")) {
            predicateBuilder.eq("unitNature",dataUnitPageVo.getUnitType());
        }
        if (!dataUnitPageVo.getUnitTerritoryId().equals("0")) {
            predicateBuilder.eq("unitArea",dataUnitPageVo.getUnitTerritoryId());
        }
        List<DataUnit> dataUnitList = dataUnitDao.findAll(predicateBuilder.build());
        dataUnitShowVo.setDataUnitList(dataUnitList);
        dataUnitShowVo.setTotalPage(dataUnitList.size());
        return dataUnitShowVo;
    }

    @Override
    public void addUnit(DataUnitAddVo dataUnitAddVo) {
        DataUnit dataUnit = new DataUnit();
        if (dataUnitAddVo.getInsertType() == 2) { // 修改
            Optional<DataUnit> byId = dataUnitDao.findById(dataUnitAddVo.getUnitId());
            if (!byId.isPresent()) {
                throw new BizException("单位不存在!");
            }
            dataUnit = byId.get();
            BeanUtils.copyProperties(dataUnitAddVo,dataUnit);
            if (dataUnitAddVo.getIsManageNode()) { // 如果作为管理单位,就清空委托单位
                dataUnit.setUnitClient(null);
                dataUnit.setUnitClientId(null);
                dataUnit.setIsClient(false);
            }
            if (dataUnitAddVo.getIsClient()) { // 如果作为委托单位,就设置管理单位为false
                dataUnit.setUnitClient(dataUnitAddVo.getUnitClient());
                dataUnit.setUnitClientId(dataUnitAddVo.getUnitClientId());
                dataUnit.setIsManageNode(false);
            }
        } else { // 新增
            // 判断信用代码证是否已经存在
            List<DataUnit> allByUnitId = dataUnitDao.findAllByUnitId(dataUnitAddVo.getUnitId());
            if (allByUnitId != null && allByUnitId.size() > 0) {
                throw new BizException("当前信用代码已存在,请重新输入!");
            }

            BeanUtils.copyProperties(dataUnitAddVo,dataUnit);

            // 新增单位在单位依据中间表中加数据
            List<ConfClassifiedBasis> all = confClassifiedBasisDao.findAll();
            List<UnitBasis> unitBasisList = new ArrayList<>();
            UnitBasis unitBasis;
            for (ConfClassifiedBasis confClassifiedBasis : all) {
                unitBasis = new UnitBasis();
                unitBasis.setId(idWorker.nextId() + "");
                unitBasis.setUnitId(dataUnit.getUnitId());
                unitBasis.setBasisId(confClassifiedBasis.getId());
                unitBasis.setIsCheck(0);
                unitBasisList.add(unitBasis);
            }
            unitBasisDao.saveAll(unitBasisList);
        }
        dataUnitDao.save(dataUnit);
    }

    @Override
    public void clientUnit(DataUnitClientVo dataUnitClientVo) {
        if (dataUnitClientVo.getDataUnitList() != null && dataUnitClientVo.getDataUnitList().size() > 0) {
            if (StringUtils.isEmpty(dataUnitClientVo.getUnitClientId())) {
                throw new BizException("没有可以委托的单位!");
            }
            PredicateBuilder<DataUnit> predicateBuilder = Specifications.and();
            List<DataUnit> dataUnitList = dataUnitClientVo.getDataUnitList();
            List<String> unitIdList = dataUnitList.stream()
                    .map(DataUnit::getUnitId)
                    .collect(Collectors.toList());
            predicateBuilder.in("unitId",unitIdList);
            List<DataUnit> unitDaoAll = dataUnitDao.findAll(predicateBuilder.build());
            for (DataUnit dataUnit : unitDaoAll) {
                dataUnit.setUnitClientId(dataUnitClientVo.getUnitClientId());
                dataUnit.setUnitClient(dataUnitClientVo.getUnitClient());
                dataUnit.setIsClient(true);
            }
            dataUnitDao.saveAll(unitDaoAll);
        } else {
            throw new BizException("没有选择被委托单位!");
        }
    }

    @Override
    public UnitClientVo findClientUnit(ClientVo clientVo) {
        UnitClientVo unitClientVo = new UnitClientVo();
        Optional<ConfUnitNode> byId = confUnitNodeDao.findById(clientVo.getNodeId());
        if (!byId.isPresent()) {
            throw new BizException("节点不存在!");
        }
        // 要设置委托单位的当前节点
        ConfUnitNode confUnitNode = byId.get();
        // .0.1.2.
        String nodeRelationship = confUnitNode.getNodeRelationship();
        // 节点路径
        List<String> stringList = Arrays.asList(nodeRelationship.split("\\."));
        // 节点过滤
        stringList = stringList.stream().filter(StringUtils::isNotEmpty).collect(Collectors.toList());
        if (stringList.size() <= 0) {
            return unitClientVo;
        }
        PredicateBuilder<ConfUnitNode> predicateBuilder = Specifications.and();
        predicateBuilder.in("id",stringList);
        // 所有可以找委托单位的节点
        List<ConfUnitNode> all = confUnitNodeDao.findAll(predicateBuilder.build());
        // 反向循环节点路径找最近的管理单位
        if (stringList.size() > 0) {
            for (int i = all.size()-1; i >= 0; i--) {
                if (all.get(i).getUnitManagementId() != null) {
                    unitClientVo.setUnitClientId(all.get(i).getUnitManagementId());
                    unitClientVo.setUnitClient(all.get(i).getUnitManagement());
                    break;
                }
            }
        }
        return unitClientVo;
    }

    @Override
    public List<ConfUnitNode> findDataUnitTree() {
        // 查询所有单位
        List<DataUnit> dataUnitList = dataUnitDao.findAll();
        // 查询所有单位节点树
        List<ConfUnitNode> confUnitNodeList = confUnitNodeDao.findAll();
        List<ConfUnitNode> newConfUnitNodeList = new ArrayList<>();
        Map<String, List<DataUnit>> collect = dataUnitList.stream().collect(Collectors.groupingBy(DataUnitServiceImpl::fetchGroupKey));
        Map<String, List<DataUnit>> collect3 = dataUnitList.stream().collect(Collectors.groupingBy(DataUnit::getUnitArea));
        ConfUnitNode confUnitNode;
        // 将所有单位变成单位节点树
        for (ConfUnitNode unitNode : confUnitNodeList) {
            if (unitNode.getIsUnitNode()) { // 是单位节点就查询单位
                List<DataUnit> orDefault;
                if (unitNode.getUnitType().equals("全部")) {
                    orDefault = collect3.getOrDefault(unitNode.getUnitTerritoryId(), new ArrayList<>());
                } else {
                    orDefault = collect.getOrDefault(unitNode.getUnitTerritoryId() + "_" + unitNode.getUnitType(), new ArrayList<>());
                }
                for (DataUnit dataUnit : orDefault) {
                    confUnitNode = new ConfUnitNode();
                    confUnitNode.setId(dataUnit.getUnitId());
                    confUnitNode.setParentId(unitNode.getId());
                    confUnitNode.setNodeName(dataUnit.getUnitName());
                    newConfUnitNodeList.add(confUnitNode);
                }
            }
        }
        confUnitNodeList.addAll(newConfUnitNodeList);
        //获取父节点
        List<ConfUnitNode> collect2 = confUnitNodeList.stream().filter(m -> unitFirstNode.equals(m.getParentId())).map(
                (m) -> {
                    m.setChildNodes(getChildrenList(m, confUnitNodeList));
                    return m;
                }
        ).collect(Collectors.toList());

        // 获取顶级节点
        List<ConfUnitNode> firstNode = confUnitNodeList.stream()
                .filter(confUnitNode2 -> confUnitNode2.getId().equals(unitFirstNode))
                .collect(Collectors.toList());
        if (firstNode.size() > 0) {
            // 顶级节点
            ConfUnitNode confFirstUnitNode = firstNode.get(0);
            confFirstUnitNode.setChildNodes(collect2);
            List<ConfUnitNode> newTree = new ArrayList<>();
            newTree.add(confFirstUnitNode);
            return newTree;
        } else {
            throw new BizException("顶级节点不存在!");
        }
    }

    @Override
    public List<DataUnit> findAllManagementUnit() {
        PredicateBuilder<DataUnit> predicateBuilder = Specifications.and();
        predicateBuilder.eq("isClient",true);
        predicateBuilder.eq("isManageNode",false);
        predicateBuilder.eq("unitClientId", MyThreadLocal.get().getUnitId());

        PredicateBuilder<DataUnit> predicateBuilder2 = Specifications.and();
        predicateBuilder2.eq("unitId",MyThreadLocal.get().getUnitId());
        List<DataUnit> all = dataUnitDao.findAll(predicateBuilder.build().or(predicateBuilder2.build()));
        return all;
    }

    // 拼接单位属地id+单位类型
    private static String fetchGroupKey(DataUnit confUnitNode) {
        return confUnitNode.getUnitArea() + "_" + confUnitNode.getUnitNature();
    }

    /**
     * 获取子节点列表
     *
     * @param tree
     * @param list
     * @return
     */
    public static List<ConfUnitNode> getChildrenList(ConfUnitNode tree, List<ConfUnitNode> list) {
        List<ConfUnitNode> children = list.stream().filter(item -> Objects.equals(item.getParentId(), tree.getId())).map(
                (item) -> {
                    item.setChildNodes(getChildrenList(item, list));
                    return item;
                }
        ).collect(Collectors.toList());
        return children;
    }
}
