package com.tykj.zspdemo.jwt;

import com.tykj.zspdemo.security.MyUserDetailsServiceImpl;
import com.tykj.zspdemo.security.SecurityUser;
import io.jsonwebtoken.*;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.Base64;
import java.util.Date;

@Component
public class JwtTokenProvider {


    @Resource
    private MyUserDetailsServiceImpl myUserDetailsService;

    private String secretKey;

    @PostConstruct
    protected void init() {
        secretKey = Base64.getEncoder().encodeToString("pkslow.key".getBytes());
    }

    /**
     * 根据用户名生成token
     * @param username
     * @return
     */
    public String createToken(String username) {

        Claims claims = Jwts.claims().setSubject(username);
        claims.put("username", username);
        Date start = new Date();
        //7小时有效时间
        long currentTime = System.currentTimeMillis() + 60 * 60 * 1000*24;
        Date end = new Date(currentTime);
        return Jwts.builder()
                .setClaims(claims)
                .setIssuedAt(start)
                .setExpiration(end)
                .signWith(SignatureAlgorithm.HS256, secretKey)
                .compact();
    }

    public Authentication getAuthentication(String token) {
        SecurityUser securityUser = (SecurityUser) myUserDetailsService.loadUserByUsername(getUsername(token));
        return new UsernamePasswordAuthenticationToken(securityUser, "", securityUser.getAuthorities());
    }

    public String getUsername(String token) {
        Claims body = Jwts.parser().setSigningKey(secretKey).parseClaimsJws(token).getBody();
        return (String) body.get("username");

    }

    public Claims getClaims(String token) {
        return Jwts.parser().setSigningKey(secretKey).parseClaimsJws(token).getBody();
    }

    /**
     * 解析token
     * @param req 请求流
     * @return token字符串
     */
    public String resolveToken(HttpServletRequest req) {
        String bearerToken = req.getHeader("Authorization");
        if (bearerToken != null && bearerToken.startsWith("Bearer ")) {
            return bearerToken.substring(7);
        }
        return null;
    }

    /**
     * 验证token
     * @param token token
     * @return 返回true或者false
     */
    public boolean validateToken(String token) {
        try {
            Jws<Claims> claims = Jwts.parser().setSigningKey(secretKey).parseClaimsJws(token);
            if (claims.getBody()==null) {
                return false;
            }
            return true;
        } catch (JwtException | IllegalArgumentException e) {
            throw new InvalidJwtAuthenticationException("Expired or invalid JWT token");
        }
    }

}
