package com.zjty.efs.ftp.service.impl;

import com.zjty.efs.ftp.dao.DownLoadCountDao;
import com.zjty.efs.ftp.entity.DownLoadCount;
import com.zjty.efs.ftp.service.FileDownLoadService;
import lombok.extern.slf4j.Slf4j;
import org.apache.catalina.connector.ClientAbortException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;

import static javax.xml.transform.OutputKeys.ENCODING;

@Service
@Slf4j
public class FileDownLoadServiceImpl implements FileDownLoadService {

    @Value("${file.address}")
    private String fileAddress;

    @Autowired
    private DownLoadCountDao countDao;

    /**
     * 下载服务器已存在的文件
     *
     * @param httpServletRequest
     *            请求对象
     * @param response
     *            响应对象
     * @param fileName
     *            文件名称
     */
    public void fileDownLoad(String fileName, DownLoadCount downLoadCount, HttpServletResponse response, HttpServletRequest httpServletRequest) {
        if(fileName != null){
            response.setHeader("content-type", "application/octet-stream");
            response.setContentType("application/octet-stream");
            try {
                response.setHeader("Content-Disposition", "attachment;filename="+new String(fileName.getBytes("utf-8"),"utf-8"));
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
            String path = fileAddress + "/" + fileName;
            File file = new File(path);
            long fileLength = file.length();
            response.setHeader("Content-Length", fileLength + "");
            //如果文件存在，返回文件，文件不存在，返回文件不存在提示
            if(fileLength != 0 ){
                setFileDownloadHeader(httpServletRequest,response,fileName);
                OutputStream os = null;
                InputStream is = null;
                BufferedInputStream bs = null;
                byte[] buffer = new byte[1024];
                try {
                    is = new FileInputStream(file);
                    bs = new BufferedInputStream(is);
                    os = response.getOutputStream();
                    int i = bs.read(buffer);
                    while (i != -1) {
                        os.write(buffer, 0, i);
                        i = bs.read(buffer);
                    }
                    os.flush();
                    //统计下载次数
                    int count = 1;
                    if(downLoadCount.getCount() != null){
                        count = downLoadCount.getCount() + 1;
                    }
                    downLoadCount.setCount(count);
                    countDao.save(downLoadCount);
                } catch (FileNotFoundException e) {
                    log.error(file.getAbsolutePath() + "文件不存在");
                    e.printStackTrace();
                } catch (IOException e) {
                    log.error(file.getAbsolutePath() + "文件读取异常");
                    e.printStackTrace();
                } finally {
                    try {
                        bs.close();
                        is.close();
                        os.close();
                    } catch (IOException e) {
                        log.error("用户终止下载");
                        //e.printStackTrace();
                    }
                }
            }else {
                log.info(path + "该文件不存在");
            }
        }else {
            log.error("请求的文件名为空");
        }
    }

    /**
     * 下载服务器已存在的文件
     *
     * @param request
     *            请求对象
     * @param response
     *            响应对象
     * @param fileName
     *            文件名称(绝对)
     */
    public void download(String fileName, HttpServletResponse response, HttpServletRequest request) {
        File proposeFile = new File(fileAddress + "/" + fileName);
        log.info("下载文件路径：" + proposeFile.getPath());
        setFileDownloadHeader(request,response,fileName);
        InputStream inputStream = null;
        OutputStream bufferOut = null;
        try {
            // 设置响应报头
            long fSize = proposeFile.length();
            response.setContentType(request.getServletContext().getMimeType(fileName));
            // Content-Disposition: attachment; filename=WebGoat-OWASP_Developer-5.2.zip
            response.setHeader("Content-Disposition", "attachment;filename="+new String(fileName.getBytes("utf-8"),"utf-8"));
            // Accept-Ranges: bytes
            response.setHeader("Accept-Ranges", "bytes");
            long pos = 0, last = fSize - 1, sum = 0;//pos开始读取位置;  last最后读取位置;  sum记录总共已经读取了多少字节
            if (null != request.getHeader("Range")) {
                // 断点续传
                response.setStatus(HttpServletResponse.SC_PARTIAL_CONTENT);
                try {
                    // 情景一：RANGE: bytes=2000070- 情景二：RANGE: bytes=2000070-2000970
                    String numRang = request.getHeader("Range").replaceAll("bytes=", "");
                    String[] strRange = numRang.split("-");
                    if (strRange.length == 2) {
                        pos = Long.parseLong(strRange[0].trim());
                        last = Long.parseLong(strRange[1].trim());
                    } else {
                        pos = Long.parseLong(numRang.replaceAll("-", "").trim());
                    }
                } catch (NumberFormatException e) {
                    log.error(request.getHeader("Range") + " is not Number!");
                    pos = 0;
                }
            }
            long rangLength = last - pos + 1;// 总共需要读取的字节
            // Content-Range: bytes 10-1033/304974592
            String contentRange = new StringBuffer("bytes ").append(pos).append("-").append(last).append("/").append(fSize).toString();
            response.setHeader("Content-Range", contentRange);
            // Content-Length: 1024
            response.addHeader("Content-Length", String.valueOf(rangLength));

            // 跳过已经下载的部分，进行后续下载
            bufferOut = new BufferedOutputStream(response.getOutputStream());
            inputStream = new BufferedInputStream(new FileInputStream(proposeFile));
            inputStream.skip(pos);
            byte[] buffer = new byte[1024];
            int length = 0;
            while (sum < rangLength) {
                length = inputStream.read(buffer, 0, ((rangLength - sum) <= buffer.length ? ((int) (rangLength - sum)) : buffer.length));
                sum = sum + length;
                bufferOut.write(buffer, 0, length);
            }
        } catch (Throwable e) {
            if (e instanceof ClientAbortException) {
                // 浏览器点击取消
                log.info("用户取消下载!");
            } else {
                log.error("下载文件失败....");
                e.printStackTrace();
            }
        } finally {
            try {
                if (bufferOut != null) {
                    bufferOut.close();
                }
                if (inputStream != null) {
                    inputStream.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 解决中文名称
     * @param request
     * @param response
     * @param fileName
     */
    public void setFileDownloadHeader(HttpServletRequest request, HttpServletResponse response, String fileName) {
        try {
            //中文文件名支持
            String encodedfileName;
            String agent = request.getHeader("USER-AGENT");
            if (null != agent && agent.contains("MSIE")) {//IE
                encodedfileName = java.net.URLEncoder.encode(fileName, "UTF-8");
            } else if (null != agent && agent.contains("Mozilla")) {
                encodedfileName = new String(fileName.getBytes("UTF-8"), "iso-8859-1");
            } else {
                encodedfileName = java.net.URLEncoder.encode(fileName, "UTF-8");
            }
            response.setHeader("Content-Disposition", "attachment; filename=\"" + encodedfileName + "\"");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
    }

}
