package com.zjty.efs.bus.websocket;

import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.websocket.*;
import javax.websocket.server.PathParam;
import javax.websocket.server.ServerEndpoint;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicInteger;

@ServerEndpoint(value = "/webServer/{userId}")
@Component
@Slf4j
public class WebSocketServer {
    @PostConstruct
    public void init() {
        log.info("websocket 加载");
    }
    private static final AtomicInteger OnlineCount = new AtomicInteger(0);
    private static Map<Session,String> map = new HashMap<>();


    /**
     * 连接建立成功调用的方法
     */
    @OnOpen
    public void onOpen(Session session,@PathParam("userId")String userId) {
        map.put(session,userId);
        int cnt = OnlineCount.incrementAndGet(); // 在线数加1
        log.info("有连接加入，当前连接数为：{}", cnt);
        SendMessage(session, "连接成功");
    }

    /**
     * 连接关闭调用的方法
     */
    @OnClose
    public void onClose(Session session) {
        map.remove(session);
        int cnt = OnlineCount.decrementAndGet();
        log.info("有连接关闭，当前连接数为：{}", cnt);
    }

    /**
     * 收到客户端消息后调用的方法
     *
     * @param message
     *            客户端发送过来的消息
     */
    @OnMessage
    public void onMessage(String message, Session session) {
        log.info("来自客户端的消息：{}",message);
        SendMessage(session, "收到消息，消息内容："+message);

    }

    /**
     * 出现错误
     * @param session
     * @param error
     */
    @OnError
    public void onError(Session session, Throwable error) {
        log.error("发生错误：{}，Session ID： {}",error.getMessage(),session.getId());
        error.printStackTrace();
    }

    /**
     * 发送消息，实践表明，每次浏览器刷新，session会发生变化。
     * @param session
     * @param message
     */
    private static void SendMessage(Session session, String message) {
        try {
            session.getBasicRemote().sendText(String.format("%s (From Server，Session ID=%s)",message,session.getId()));
        } catch (IOException e) {
            log.error("发送消息出错：{}", e.getMessage());
            e.printStackTrace();
        }
    }


    /**
     * 给所有的Session推送信息
     * @param message 需要往客户端推送的信息
     * @param userId 客户端登录用户的编号
     */

    public void sendData(String userId,String message){
        log.info("websocket推送信息: "+message);
        List<Session> list = getKey(map, userId);
        if (!list.isEmpty()){
            for(Session session:list){
                if(session.isOpen()){
                    SendMessage(session, message);
                }
            }

        }


    }

    //根据map的value获取map的key
    private static List<Session> getKey(Map<Session,String> map, String value){
        List<Session> sessionList = new ArrayList<>();
        for (Map.Entry<Session, String> entry : map.entrySet()) {
            if(value.equals(entry.getValue())){
                sessionList.add(entry.getKey());
            }
        }
        return sessionList;
    }



}
