package com.zjty.tynotes.job.basic.controller;

import com.zjty.tynotes.job.basic.entity.database.Work;
import com.zjty.tynotes.job.basic.entity.request.UserRo;
import com.zjty.tynotes.job.basic.entity.request.WorkRo;
import com.zjty.tynotes.job.basic.entity.response.JobResponse;
import com.zjty.tynotes.job.basic.entity.response.WorkVo;
import com.zjty.tynotes.job.basic.service.TransHelper;
import com.zjty.tynotes.job.basic.service.WorkService;
import com.zjty.tynotes.job.business.service.AlterTaskService;
import com.zjty.tynotes.job.business.service.NoteMessageService;
import com.zjty.tynotes.job.common.Action;
import com.zjty.tynotes.job.status.service.BusinessTreeManagement;
import com.zjty.tynotes.misc.config.AutoDocument;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import static com.zjty.tynotes.job.common.Action.PUBLISH_WORK;
import static java.util.Objects.nonNull;
import static org.springframework.http.ResponseEntity.ok;

/**
 * <p>Description : tynotes
 * <p>Date : 2019/4/22 15:02
 * <p>@author : C
 */
@AutoDocument
@Api(tags = "工作记录相关接口", description = "工作记录模块")
@RestController
@RequestMapping("/job/work")
@Slf4j
public class WorkController {

    private final
    WorkService workService;

    private final
    TransHelper transHelper;

    private final
    NoteMessageService noteMessageService;

    @Autowired
    private AlterTaskService alterTaskService;
    @Autowired
    private BusinessTreeManagement businessTreeManagement;

    @Autowired
    public WorkController(WorkService workService, TransHelper transHelper, NoteMessageService noteMessageService) {
        this.workService = workService;
        this.transHelper = transHelper;
        this.noteMessageService = noteMessageService;
    }

    @GetMapping(value = "/")
    @ApiOperation(value = "查询所有工作.")
    public ResponseEntity<List<WorkVo>> findAll() {
        List<WorkVo> workVoList = workService.findAll().stream()
                .map(transHelper::transToVo)
                .collect(Collectors.toList());
        return ok(workVoList);
    }

    @GetMapping(value = "/{workId}")
    @ApiOperation(value = "根据id查询工作.", notes = "在路径中指定要查询的工作id.")
    public ResponseEntity<WorkVo> findById(@PathVariable String workId) {
        WorkVo workVo = transHelper.transToVo(workService.findById(workId));
        return ok(workVo);

    }


    @PostMapping(value = "/")
    @ApiOperation(value = "新增工作.", notes = "新增不可在数据中附带id.成功时返回新增数据保存的id.")
    public ResponseEntity<JobResponse> add(@RequestBody WorkRo workRo) {
        String userId = workRo.getUserId();
        String remarks = workRo.getRemarks();
        Work work = workRo.getWork();
        String workId = workService.add(work);
        Boolean isFormal = work.getIsFormal();
        if (nonNull(isFormal) && isFormal) {
            noteMessageService.saveNoteMessage(userId, workId, PUBLISH_WORK, remarks);
        }
        return ok(new JobResponse(workId));
    }

    @PutMapping(value = "/")
    @ApiOperation(value = "更新工作.", notes = "修改必须在数据中附带id.")
    public ResponseEntity<JobResponse> modify(@RequestBody WorkRo workRo) {
        String userId = workRo.getUserId();
        String remarks = workRo.getRemarks();
        Work newWork = workRo.getWork();
        Work oldWork = workService.findById(newWork.getId());
        String workId = workService.modify(newWork);
        noteMessageService.saveNoteMessage(userId, workId, newWork, oldWork, remarks);
        return ok(new JobResponse(workId));
    }

    @GetMapping(value = "/draft/{userId}")
    @ApiOperation(value = "根据当前用户id查询其拥有的草稿.")
    public ResponseEntity<List<WorkVo>> add(@PathVariable String userId) {
        return ok(workService.findAll().stream()
                .filter(work1 -> !work1.getIsFormal())
                .filter(work -> work.getPublisher().equals(userId))
                .map(transHelper::transToVo)
                .collect(Collectors.toList()));
    }

    @DeleteMapping(value = "/{workId}")
    @ApiOperation(value = "根据id删除工作.", notes = "在路径中指定要删除的工作id.")
    public ResponseEntity<JobResponse> delete(@PathVariable String workId) {
        workService.delete(workId);
        return ok(new JobResponse(workId));
    }

    @PutMapping(value = "/status")
    @ApiOperation(value = "修改工作状态")
    public ResponseEntity alterTaskStatus(@RequestParam String taskId,
                                          @RequestParam String userId,
                                          @RequestParam String status,
                                          @RequestParam String msg) {
        switch (status) {
            case "underway":
                alterTaskService.alterTaskStatus(taskId, status);
                businessTreeManagement.saveAction(userId, taskId, Action.BACK_WORK, new Date(), msg);
                break;
            case "finished":
                alterTaskService.alterTaskStatus(taskId, status);
                businessTreeManagement.saveAction(userId, taskId, Action.FINISH_WORK, new Date(), msg);
                break;
            case "committed":
                alterTaskService.alterTaskStatus(taskId, status);
                businessTreeManagement.saveAction(userId, taskId, Action.COMMIT_WORK, new Date(), msg);
                break;
            default:
                log.warn("[job] 修改任务状态传入的参数有误，传入值为：{}", status);
                break;
        }
        return ok("success");
    }

    @PostMapping(value = "/updateChecker")
    @ApiOperation(value = "修改审核者")
    public ResponseEntity updateChecker(@RequestBody UserRo userRo) {
        alterTaskService.updateChecker(userRo);
        businessTreeManagement.saveAction(userRo.getOperatorId(), userRo.getWorkId(), Action.ALTER_CHECKER,
                new Date(), "");
        return ok("success");
    }

    @PostMapping(value = "/updateFlower")
    @ApiOperation(value = "修改关注者")
    public ResponseEntity updateFlower(@RequestBody UserRo userRo) {
        alterTaskService.updateFlower(userRo);
        businessTreeManagement.saveAction(userRo.getOperatorId(), userRo.getWorkId(), Action.ALTER_FOLLOWER,
                new Date(), "");
        return ok("success");
    }


}
