package com.zjty.tynotes.job.basic.service.impl;

import com.google.common.collect.Lists;
import com.zjty.tynotes.job.basic.entity.database.Attachment;
import com.zjty.tynotes.job.basic.entity.database.Work;
import com.zjty.tynotes.job.basic.repository.AttachmentRepository;
import com.zjty.tynotes.job.basic.service.AttachmentService;
import com.zjty.tynotes.job.basic.service.WorkService;
import com.zjty.tynotes.job.common.exception.BadRequestException;
import com.zjty.tynotes.job.common.exception.NotFoundException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Optional;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;

/**
 * <p>Description : note
 * <p>Date : 2019/4/24 14:11
 * <p>@author : C
 * 接口定义说明见↓
 *
 * @see AttachmentService
 */
@Slf4j
@Service
public class AttachmentServiceImpl implements AttachmentService {

    private final
    AttachmentRepository attachmentRepository;

    private final
    WorkService workService;

    @Autowired
    public AttachmentServiceImpl(AttachmentRepository attachmentRepository, WorkService workService) {
        this.attachmentRepository = attachmentRepository;
        this.workService = workService;
    }

    @Override
    public String add(Attachment attachment) {
        String msg;
        //新增数据不可附带id.
        Boolean idMustNull = isNull(attachment.getId());
        if (idMustNull) {
            //更新时间
            attachment.setTime(new Date());
            String saveId = attachmentRepository.save(attachment).getId();
            msg = String.format("[job] 新增了id为 %s 的数据.", saveId);
            log.info(msg);
            return saveId;
        } else {
            msg = "[job] 新增数据不可附带id.";
            log.error(msg);
            throw new BadRequestException(msg);
        }
    }

    @Override
    public String add(Attachment attachment, String workId) {
        //新增数据不可附带id.
        Boolean idMustNull = isNull(attachment.getId());
        if (idMustNull) {
            Boolean workMustExist = workService.idExists(workId);
            if (workMustExist) {
                String saveId = attachmentRepository.save(attachment).getId();
                Work work = workService.findById(workId);
                if (isNull(work.getAttachment())) {
                    work.setAttachment(Lists.newArrayList());
                }
                work.getAttachment().add(saveId);
                workService.modify(work);
                log.info(String.format("[job] id为 %s 的附件已添加到id为 %s 的所属工作中.", saveId, workId));
                return saveId;
            } else {
                String msg = "[job] 指定id的所属工作不存在.";
                log.error(msg);
                throw new NotFoundException(msg);
            }
        } else {
            String msg = "[job] 新增数据不可附带id.";
            log.error(msg);
            throw new BadRequestException(msg);
        }
    }

    @Override
    public String add(List<Attachment> attachments, String workId) {
        if (attachments.size() == 0) {
            String msg = "[Attachment] 没有附件";
            log.error(msg);
            throw new BadRequestException(msg);
        }
        for (Attachment attachment : attachments){
            Boolean idMustNull = isNull(attachment.getId());
        if (idMustNull) {
            Boolean workMustExist = workService.idExists(workId);
            if (workMustExist) {
                String saveId = attachmentRepository.save(attachment).getId();
                Work work = workService.findById(workId);
                if (isNull(work.getAttachment())) {
                    work.setAttachment(Lists.newArrayList());
                }
                work.getAttachment().add(saveId);
                workService.modify(work);
            } else {
                String msg = "[job] 指定id的所属工作不存在.";
                log.error(msg);
                throw new NotFoundException(msg);
            }
        } else {
            String msg = "[job] 新增数据不可附带id.";
            log.error(msg);
            throw new BadRequestException(msg);
        }
    }
    return "添加附件成功";
    }

    @Override
    public String modify(Attachment attachment) {
        Boolean idExist = nonNull(attachment.getId());
        if (idExist) {
            if (attachmentRepository.existsById(attachment.getId())) {
                attachment.setTime(new Date());
                String modifyId = attachmentRepository.save(attachment).getId();
                log.info(String.format("[job] 修改了id为 %s 的数据.", modifyId));
                return modifyId;
            } else {
                String msg = "[job] 指定id的数据不存在.";
                log.error(msg);
                throw new NotFoundException(msg);
            }
        } else {
            String msg = "[job] 修改必须在数据中附带id.";
            log.error(msg);
            throw new BadRequestException(msg);
        }
    }

    @Override
    public Attachment findById(String attachmentId) {
        Optional<Attachment> workOptional = attachmentRepository.findById(attachmentId);
        return workOptional.orElseThrow(() -> new BadRequestException(String.format("[job] id为 %s 的数据不存在.", attachmentId)));
    }

    @Override
    public List<Attachment> findAll() {
        return attachmentRepository.findAll();
    }

    @Override
    public List<Attachment> findAll(List<String> ids) {
        return Lists.newArrayList(attachmentRepository.findAllById(ids));
    }

    @Override
    public void delete(String id) {

    }

    @Override
    public void deleteAll() {
        attachmentRepository.deleteAll();
    }

}
